% this handles the stochastic-gradient-descent CMA algorithms for the
% bergulator, and is called by ui_traj.m
%
% you can create a .mex version by typing "mex -O berg_algs.c" at the 
% system prompt
%
% WARNING: you must ensure that the input sequence "r" has length of 
% at least N_f+(1+is_FSE)*N.  the .mex version will NOT check this,
% and could read into invalid memory locations leading to "misconvergence"
% near the end of the adaptation record!
 
% Copyright 1997-1998 Phil Schniter

function [f,y,srcme] = berg_algs(N,N_f,D_f,D_e,is_FSE,kappa,mu,r,f_init,...
				 alg, misc)
 
 if size(r,2) < N_f+(1+is_FSE)*N, 
   error('r must be a row vector of length >= N_f+(1+is_FSE)*N'); 
 end;
 f_cur = f_init;
 if alg == 1,           % straight-up CMA
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   D_u = misc;				       % update-decimation factor
   for i = D_u:D_u:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur+mu*reg'*(y_cur*srcme_cur);
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 elseif alg == 2,       % normalized CMA (N-CMA)
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   alpha = misc;			       % regularization term
   for i = 1:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur + mu/(alpha+norm(reg)^2)*reg'*(y_cur*srcme_cur);
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 elseif alg == 3,       % signed-error CMA (SE-CMA)
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   for i = 1:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur + mu*reg'*...
       (sign(real(y_cur)*srcme_cur)+j*sign(imag(y_cur)*srcme_cur));
     %f_cur = f_cur + mu*reg'*(y_cur*srcme_cur)/abs(y_cur*srcme_cur);
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 elseif alg == 4,       % signed-regressor CMA (SR-CMA)
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   for i = 1:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur+mu*(sign(real(reg))+j*sign(imag(reg)))'*(y_cur*srcme_cur);
     %f_cur = f_cur + mu*(reg./abs(reg))'*(y_cur*srcme_cur);
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 elseif alg == 5,       % signed-error, signed-regressor CMA (SS-CMA)
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   for i = 1:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur + mu*(sign(real(reg))+j*sign(imag(reg)))'*...
       (sign(real(y_cur)*srcme_cur)+j*sign(imag(y_cur)*srcme_cur));
     %f_cur = f_cur+mu*(reg./abs(reg))'*(y_cur*srcme_cur)/abs(y_cur*srcme_cur);
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 elseif alg == 6,       % dithered signed-error CMA (DSE-CMA)
   alpha = misc;			       % dither amplitude
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   dith_r = alpha*(2*rand(1,N)-1);	       % real dither
   if ~isreal(r),
     dith_i = alpha*(2*rand(1,N)-1);	       % imag dither
   end;
   mu = mu*alpha; 			       % adjust step size w.r.t. alpha
   for i = 1:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur + mu*reg'*sign(real(y_cur)*srcme_cur+dith_r(i));
     if ~isreal(r),
       f_cur = f_cur + j*mu*reg'*sign(imag(y_cur)*srcme_cur+dith_i(i));
     end;
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 elseif alg == 7,       % weerackody's signed-godard algorithm (WSGA)
   f = zeros(N_f,floor((N-1)/D_f)+1);          % equalizer history
   y = zeros(1,floor((N-1)/D_e)+1);            % soft decisions
   srcme = zeros(1,floor((N-1)/D_e)+1);        % sqrt(CM) error history
   for i = 1:N,
     reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
     y_cur = reg*f_cur;
     srcme_cur = kappa-abs(y_cur)^2;
     f_cur = f_cur + mu*reg'*sign(kappa-abs(real(y_cur))-abs(imag(y_cur)))*...
       (sign(real(y_cur))+j*sign(imag(y_cur)));
     f(:,floor((i-1)/D_f)+1) = f_cur;
     y(floor((i-1)/D_e)+1) = y_cur;
     srcme(floor((i-1)/D_e)+1) = srcme_cur;
   end
 else
   error('That algorithm is not supported by berg_algs.m');
 end;
