/******************************************************************************
 *									      *
 * berg_algs.c - adaptation routines for the BERGulator (called by ui_traj.m) *
 *									      *
 * written by: 	Phil "Bert" Schniter 					      *
 *		Blind Equalization Research Group 			      *
 * 		Cornell University					      *
 *		9/18/97							      *
 *									      *
 * Copyright 1997-1998 Phil Schniter					      *
 *									      *
 ******************************************************************************/

#include "mex.h"
#include <math.h>

/* to be consistent with the user interface algorithm ordering... */
#define	 CMA 	1
#define	 NCMA 	2
#define	 SECMA  3
#define	 SRCMA  4
#define	 SSCMA  5
#define	 DSECMA 6
#define	 WSGA   7
#define	 CMAGD  8


/******************** 
 * standard CMA 2-2 *
 ********************
 *
 * the CMA 2-2 routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * f_cur = f_init;
 * for i = D_u:D_u:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur+mu*reg'*(y_cur*srcme_cur);
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void cma(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y, int D_u)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu_yr_srcme, mu_yi_srcme;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer */
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
    }

  /* adaptation routine */
  reg_ind = N_f-1;
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += (1+is_FSE)*D_u;	/* update regressor index */

      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	
      mu_yr_srcme = mu*yr_cur*srcme_cur;
      mu_yi_srcme = mu*yi_cur*srcme_cur;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k]*mu_yr_srcme + ri[reg_ind-k]*mu_yi_srcme;
 	fi_cur[k] += rr[reg_ind-k]*mu_yi_srcme - ri[reg_ind-k]*mu_yr_srcme;
      }
      i += D_u;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued... */
    for (i=0; i<N; ){
      reg_ind += (1+is_FSE)*D_u;	/* update regressor index */

      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */
      mu_yr_srcme = mu*yr_cur*srcme_cur;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k] * mu_yr_srcme;
      }
      i += D_u;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of cma() */




/********************** 
 * normalized CMA 2-2 *
 **********************
 *
 * the NCMA routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * f_cur = f_init;
 * alpha = 0.001;
 * for i = 1:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur + mu/(alpha+norm(reg)^2)*reg'*(y_cur*srcme_cur);
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void ncma(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y, double alpha)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu_yr_srcme, mu_yi_srcme, sqnrm_reg;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer and squared regressor norm */
  reg_ind = N_f-1;
  sqnrm_reg = 0;
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
    sqnrm_reg += rr[reg_ind-k]*rr[reg_ind-k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
      sqnrm_reg += ri[reg_ind-k]*ri[reg_ind-k];
    }

  /* adaptation routine */
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* update squared regressor norm */
      sqnrm_reg += rr[reg_ind]*rr[reg_ind] + ri[reg_ind]*ri[reg_ind]
	           - rr[reg_ind-N_f]*rr[reg_ind-N_f] 
                   - ri[reg_ind-N_f]*ri[reg_ind-N_f];
      if( is_FSE ){
        sqnrm_reg += rr[reg_ind-1]*rr[reg_ind-1] + ri[reg_ind-1]*ri[reg_ind-1]
  	             - rr[reg_ind-N_f-1]*rr[reg_ind-N_f-1]
  	             - ri[reg_ind-N_f-1]*ri[reg_ind-N_f-1];
      }
      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	
      mu_yr_srcme = mu*yr_cur*srcme_cur/(alpha + sqnrm_reg);
      mu_yi_srcme = mu*yi_cur*srcme_cur/(alpha + sqnrm_reg);

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k]*mu_yr_srcme + ri[reg_ind-k]*mu_yi_srcme;
 	fi_cur[k] += rr[reg_ind-k]*mu_yi_srcme - ri[reg_ind-k]*mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* update squared regressor norm */
      sqnrm_reg += rr[reg_ind]*rr[reg_ind] 
	           - rr[reg_ind-N_f]*rr[reg_ind-N_f];
      if( is_FSE ){
        sqnrm_reg += rr[reg_ind-1]*rr[reg_ind-1] 
  	             - rr[reg_ind-N_f-1]*rr[reg_ind-N_f-1];
      }
      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */
      mu_yr_srcme = mu*yr_cur*srcme_cur/(alpha + sqnrm_reg);

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k] * mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of ncma() */




/************************ 
 * signed-error CMA 2-2 *
 ************************
 *
 * the SE-CMA routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * f_cur = f_init;
 * for i = 1:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur + mu*reg'*...
 *      (sign(real(y_cur)*srcme_cur)+j*sign(imag(y_cur)*srcme_cur));
 *   %f_cur = f_cur + mu*reg'*(y_cur*srcme_cur)/abs(y_cur*srcme_cur);
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void secma(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu_yr_srcme, mu_yi_srcme;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer */
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
    }

  /* adaptation routine */
  reg_ind = N_f-1;
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	
      if( yr_cur*srcme_cur > 0 )
        mu_yr_srcme = mu;		/* retain only sign of real error */
      else
        mu_yr_srcme = -mu;
      if( yi_cur*srcme_cur > 0 )
        mu_yi_srcme = mu;		/* retain only sign of imag error */
      else
        mu_yi_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k]*mu_yr_srcme + ri[reg_ind-k]*mu_yi_srcme;
 	fi_cur[k] += rr[reg_ind-k]*mu_yi_srcme - ri[reg_ind-k]*mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */
      if( yr_cur*srcme_cur > 0 )
        mu_yr_srcme = mu;		/* retain only sign of error */
      else
        mu_yr_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k] * mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of secma() */




/**************************** 
 * signed-regressor CMA 2-2 *
 ****************************
 *
 * the SR-CMA routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * f_cur = f_init;
 * for i = 1:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur+mu*(sign(real(reg))+j*sign(imag(reg)))'*(y_cur*srcme_cur);
 *   %f_cur = f_cur + mu*(reg./abs(reg))'*(y_cur*srcme_cur);
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void srcma(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu_yr_srcme, mu_yi_srcme;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer */
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
    }

  /* adaptation routine */
  reg_ind = N_f-1;
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	
      mu_yr_srcme = mu*yr_cur*srcme_cur;	
      mu_yi_srcme = mu*yi_cur*srcme_cur;	

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
        if (rr[reg_ind-k] > 0){
 	  fr_cur[k] += mu_yr_srcme;
 	  fi_cur[k] += mu_yi_srcme;
     	}
 	else{
 	  fr_cur[k] -= mu_yr_srcme;
 	  fi_cur[k] -= mu_yi_srcme;
	}
        if (ri[reg_ind-k] > 0){
 	  fr_cur[k] += mu_yi_srcme;
 	  fi_cur[k] -= mu_yr_srcme;
	}
 	else{
 	  fr_cur[k] -= mu_yi_srcme;
 	  fi_cur[k] += mu_yr_srcme;
	}
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */
      mu_yr_srcme = mu*yr_cur*srcme_cur;	

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
        if (rr[reg_ind-k] > 0)
 	  fr_cur[k] += mu_yr_srcme;
 	else
 	  fr_cur[k] -= mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of srcma() */




/***************************************** 
 * signed-regressor signed-error CMA 2-2 *
 *****************************************
 *
 * the SS-CMA routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * f_cur = f_init;
 * for i = 1:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur + mu*(sign(real(reg))+j*sign(imag(reg)))'*...
 *     (sign(real(y_cur)*srcme_cur)+j*sign(imag(y_cur)*srcme_cur));
 *   %f_cur = f_cur+mu*(reg./abs(reg))'*(y_cur*srcme_cur)/abs(y_cur*srcme_cur);
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void sscma(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu_yr_srcme, mu_yi_srcme;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer */
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
    }

  /* adaptation routine */
  reg_ind = N_f-1;
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	
      if( yr_cur*srcme_cur > 0 )
        mu_yr_srcme = mu;               /* retain only sign of real error */
      else
        mu_yr_srcme = -mu;
      if( yi_cur*srcme_cur > 0 )
        mu_yi_srcme = mu;               /* retain only sign of imag error */
      else
        mu_yi_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
        if (rr[reg_ind-k] > 0){
 	  fr_cur[k] += mu_yr_srcme;
 	  fi_cur[k] += mu_yi_srcme;
     	}
 	else{
 	  fr_cur[k] -= mu_yr_srcme;
 	  fi_cur[k] -= mu_yi_srcme;
	}
        if (ri[reg_ind-k] > 0){
 	  fr_cur[k] += mu_yi_srcme;
 	  fi_cur[k] -= mu_yr_srcme;
	}
 	else{
 	  fr_cur[k] -= mu_yi_srcme;
 	  fi_cur[k] += mu_yr_srcme;
	}
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */
      if( yr_cur*srcme_cur > 0 )
        mu_yr_srcme = mu;               /* retain only sign of error */
      else
        mu_yr_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
        if (rr[reg_ind-k] > 0)
 	  fr_cur[k] += mu_yr_srcme;
 	else
 	  fr_cur[k] -= mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of sscma() */




/********************************* 
 * dithered signed-error CMA 2-2 *
 *********************************
 *
 * the DSE-CMA routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * dith_r = alpha*(2*rand(1,N)-1);
 * dith_i = alpha*(2*rand(1,N)-1);
 * mu = mu*alpha;
 * f_cur = f_init;
 * for i = 1:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur + mu*reg'*sign(real(y_cur)*srcme_cur+dith_r(i));
 *   if ~isreal(r),
 *     f_cur = f_cur + j*mu*reg'*sign(imag(y_cur)*srcme_cur+dith_i(i));
 *   end;
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void dsecma(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu_without_alpha,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y, double alpha)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu; 
  double mu_yr_srcme, mu_yi_srcme;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 
  double *dr, *di;				/* dither records */
  int 	  num_in=2, num_out=1;			/* # i/o args for "rand" */
  mxArray *prhs[2], *plhs1[1], *plhs2[1];	/* arrays of matlab arrays */
  double  rand_in[2];				/* inputs to "rand": {N,1} */


  /* first create dither signal(s)... */
  /* set inputs to "rand" function */
   prhs[0] = mxCreateDoubleMatrix(1,1,mxREAL);	/* make matlab array */
   prhs[1] = mxCreateDoubleMatrix(1,1,mxREAL);	/* make matlab array */
   rand_in[0] = N;				/* set numerical value */
   rand_in[1] = 1;				/* set numerical value */
   memcpy(mxGetPr(prhs[0]), rand_in, sizeof(double));  	/* copy values */
   memcpy(mxGetPr(prhs[1]), rand_in+1, sizeof(double));	/* copy values */
  /* call matlab's "rand" */
   mexCallMATLAB(num_out,plhs1,num_in,prhs,"rand");
   dr = mxGetPr(plhs1[0]);
   if( is_cmplx ){
     mexCallMATLAB(num_out,plhs2,num_in,prhs,"rand"); 
     di = mxGetPr(plhs2[0]);
   }
  /* normalize dither to [-alpha,alpha) */
   for (k=0; k<N; k++){ dr[k] -= 0.5; dr[k] *= 2*alpha; }
   if( is_cmplx ){
     for (k=0; k<N; k++){ di[k] -= 0.5; di[k] *= 2*alpha; }
   }
  
  /* normalize stepsize according to alpha */
  mu = alpha*mu_without_alpha;

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer */
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
    }

  /* adaptation routine */
  reg_ind = N_f-1;
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	

      /* dithered decisions */
      if( yr_cur*srcme_cur + dr[i] > 0 )
        mu_yr_srcme = mu;		/* retain only sign of real err */
      else
        mu_yr_srcme = -mu;
      if( yi_cur*srcme_cur + di[i] > 0 )
        mu_yi_srcme = mu;		/* retain only sign of imag err */
      else
        mu_yi_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k]*mu_yr_srcme + ri[reg_ind-k]*mu_yi_srcme;
 	fi_cur[k] += rr[reg_ind-k]*mu_yi_srcme - ri[reg_ind-k]*mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */

      /* dithered decision */
      if( yr_cur*srcme_cur + dr[i] > 0 )
        mu_yr_srcme = mu;		/* retain only sign of error */
      else
        mu_yr_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k] * mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of dsecma() */




/***************************************** 
 * Weekrackody's signed Godard algorithm *
 *****************************************
 *
 * the WSGA routine replaces the following matlab code:
 *
 * --------------------------------------------------
 * f_cur = f_init;
 * for i = 1:N,
 *   reg = r(1, N_f+(1+is_FSE)*i:-1:(1+is_FSE)*i+1);
 *   y_cur = reg*f_cur;
 *   srcme_cur = kappa-abs(y_cur)^2;
 *   f_cur = f_cur + mu*reg'*sign(kappa-abs(real(y_cur))-abs(imag(y_cur)))*...
 *              (sign(real(y_cur))+j*sign(imag(y_cur)));
 *   f(:,floor((i-1)/D_f)+1) = f_cur;
 *   y(floor((i-1)/D_e)+1) = y_cur;
 *   srcme(floor((i-1)/D_e)+1) = srcme_cur;
 * end
 * --------------------------------------------------
 */
void wsga(double *fr, double *fi, double *yr, double *yi, double *srcme, 
         double *rr, double *ri, double *fr_init, double *fi_init,
	 double kappa, double mu,
         int N, int N_f, int D_e, int D_f, int is_FSE, bool is_cmplx,
	 int len_f, int len_y)
{
  double *fr_cur, *fi_cur;
  double yr_cur, yi_cur, srcme_cur, mu_yr_srcme, mu_yi_srcme, sgn_yr, sgn_yi;
  int    i, k, f_ind=0, y_ind=0, reg_ind; 

  /* allocate local memory */
  fr_cur = (double *)mxCalloc(N_f,sizeof(double));
  fi_cur = (double *)mxCalloc(N_f,sizeof(double));

  /* initialize equalizer */
  for (k=0; k<N_f; k++){
    fr_cur[k] = fr_init[k];
  }
  if( is_cmplx )
    for (k=0; k<N_f; k++){
      fi_cur[k] = fi_init[k];
    }

  /* adaptation routine */
  reg_ind = N_f-1;
  if( is_cmplx ){		/* if complex-valued... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      /* calc equalizer output */
      yr_cur = 0;			
      yi_cur = 0;			
      for (k=0; k<N_f; k++){ 		
        yr_cur += rr[reg_ind-k]*fr_cur[k] - ri[reg_ind-k]*fi_cur[k];
        yi_cur += rr[reg_ind-k]*fi_cur[k] + ri[reg_ind-k]*fr_cur[k];
      }
      /* calc square-root CM error */
      srcme_cur = kappa-(yr_cur*yr_cur+yi_cur*yi_cur);	
  
      /* calculate WSGA update */ 
      if( yr_cur > 0 ) sgn_yr = 1.0; else sgn_yr = -1.0;
      if( yi_cur > 0 ) sgn_yi = 1.0; else sgn_yi = -1.0;
      if( kappa - sgn_yr*yr_cur - sgn_yi*yi_cur > 0 ){
        mu_yr_srcme = mu*sgn_yr;	
        mu_yi_srcme = mu*sgn_yi;	
      }
      else{
        mu_yr_srcme = -mu*sgn_yr;
        mu_yi_srcme = -mu*sgn_yi;	
      }
      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k]*mu_yr_srcme + ri[reg_ind-k]*mu_yi_srcme;
 	fi_cur[k] += rr[reg_ind-k]*mu_yi_srcme - ri[reg_ind-k]*mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind] = fr_cur[k];
          fi[f_ind++] = fi_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store output */
        yi[y_ind] = yi_cur;
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }
  else{				/* if real-valued (same as SE-CMA)... */
    for (i=0; i<N; ){
      reg_ind += 1+is_FSE;		/* update regressor index */

      yr_cur = 0;			
      for (k=0; k<N_f; k++){ 		/* calc equalizer output */
        yr_cur += rr[reg_ind-k]*fr_cur[k];
      }
      srcme_cur = kappa-yr_cur*yr_cur;	/* calc square-root CM error */
      if( yr_cur*srcme_cur > 0 )
        mu_yr_srcme = mu;		/* retain only sign of error */
      else
        mu_yr_srcme = -mu;

      for (k=0; k<N_f; k++){		/* update equalizer coefficients */
 	fr_cur[k] += rr[reg_ind-k] * mu_yr_srcme;
      }
      i++;				/* convenient to increment here */
      if( !(i % D_f) ) 
        for (k=0; k<N_f; k++){		/* store equalizer coefficients */
          fr[f_ind++] = fr_cur[k];
        }
      if( !(i % D_e) ){
        yr[y_ind] = yr_cur;		/* store ouput */
	srcme[y_ind++] = srcme_cur;	/* store srcme */
      }
    } 
  }

  /* store final outputs */
  for (k=0; k<N_f; k++){
    fr[N_f*(len_f-1)+k] = fr_cur[k];
  }
  yr[len_y-1] = yr_cur;
  srcme[len_y-1] = srcme_cur;
  if( is_cmplx ){
    for (k=0; k<N_f; k++){
      fi[N_f*(len_f-1)+k] = fi_cur[k];
    }
    yi[len_y-1] = yi_cur;
  }
}/* end of wsga() */




/********************** 
 * the MATLAB gateway *
 **********************/
void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
 /**********************************************************************
  * input args: {N,N_f,D_f,D_e,is_FSE,kappa,mu,r,f_init,alg_num,misc}  *
  * output args: {f,y,srcme}                                           *
  **********************************************************************/

  bool	 is_cmplx;
  int 	 N, N_f, D_e, D_f, is_FSE, len_y, len_f, alg_num, D_u;
  double kappa, mu, alpha; 
  double *rr, *ri, *fr, *fi, *yr, *yi, *srcme, *fr_init, *fi_init; 

  /* check for the proper number of arguments */
  if(nrhs!=11)
    mexErrMsgTxt("Eleven inputs required.");
  if(nlhs!=3)
    mexErrMsgTxt("Three outputs required.");

  /* check for sizes on input arguments */
  if( mxGetM(prhs[0])!=1 || mxGetN(prhs[0])!=1 || !mxIsDouble(prhs[0]) )
    mexErrMsgTxt("First argument must be a scalar: N");
  if( mxGetM(prhs[1])!=1 || mxGetN(prhs[1])!=1 || !mxIsDouble(prhs[1]) )
    mexErrMsgTxt("Second argument must be a scalar: N_f");
  if( mxGetM(prhs[2])!=1 || mxGetN(prhs[2])!=1 || !mxIsDouble(prhs[2]) )
    mexErrMsgTxt("Third argument must be a scalar: D_f");
  if( mxGetM(prhs[3])!=1 || mxGetN(prhs[3])!=1 || !mxIsDouble(prhs[3]) )
    mexErrMsgTxt("Fourth argument must be a scalar: D_e");
  if( (int)mxGetScalar(prhs[4])!=0 && (int)mxGetScalar(prhs[4])!=1 )
    mexErrMsgTxt("Fifth argument must be in set {0,1}: is_FSE");
  if( mxGetM(prhs[5])!=1 || mxGetN(prhs[5])!=1 || !mxIsDouble(prhs[5]) )
    mexErrMsgTxt("Sixth argument must be a scalar: kappa");
  if( mxGetM(prhs[6])!=1 || mxGetN(prhs[6])!=1 || !mxIsDouble(prhs[6]) )
    mexErrMsgTxt("Seventh argument must be a scalar: mu");
  if( mxGetM(prhs[7])!=1 || !mxIsDouble(prhs[7]) )
    mexErrMsgTxt("Eighth argument must be a row vector: r");
  if( mxGetN(prhs[8])!=1 || !mxIsDouble(prhs[8]) )
    mexErrMsgTxt("Ninth argument must be a column vector: f_init");
  if( (int)mxGetScalar(prhs[9])<1 || (int)mxGetScalar(prhs[9])>8 )
    mexErrMsgTxt("Tenth argument must be an integer: 1 <= alg_num <= 8");

  /* retrieve scalars */ 
  N = (int)mxGetScalar(prhs[0]);
  N_f = (int)mxGetScalar(prhs[1]);
  D_f = (int)mxGetScalar(prhs[2]);
  D_e = (int)mxGetScalar(prhs[3]);
  is_FSE = (int)mxGetScalar(prhs[4]);
  kappa = (double)mxGetScalar(prhs[5]);
  mu = (double)mxGetScalar(prhs[6]);
  is_cmplx = mxIsComplex(prhs[7]);	/* true if r is complex */
  alg_num = (int)mxGetScalar(prhs[9]);

  /* set input pointer(s) */
  rr = mxGetPr(prhs[7]);
  fr_init = mxGetPr(prhs[8]);
  if( is_cmplx ){ 
    ri = mxGetPi(prhs[7]);
    if( mxIsComplex(prhs[8]) )		/* true if f_init is complex */
      fi_init = mxGetPi(prhs[8]);
    else
      fi_init = (double *)mxCalloc(N_f,sizeof(double));
  }

  /* determine decimated lengths */
  len_y = (N-1)/D_e + 1;
  len_f = (N-1)/D_f + 1;

  /* allocate memory for outputs */
  if( is_cmplx ){ 
    plhs[0] = mxCreateDoubleMatrix(N_f,len_f,mxCOMPLEX);	/* f */
    plhs[1] = mxCreateDoubleMatrix(1,len_y,mxCOMPLEX);   	/* y */
  }
  else{
    plhs[0] = mxCreateDoubleMatrix(N_f,len_f,mxREAL);    	/* f */
    plhs[1] = mxCreateDoubleMatrix(1,len_y,mxREAL);      	/* y */
  }
  plhs[2] = mxCreateDoubleMatrix(1,len_y,mxREAL); 		/* srcme */
  
  /* set output pointers */
  fr = mxGetPr(plhs[0]); 
  yr = mxGetPr(plhs[1]); 
  srcme = mxGetPr(plhs[2]); 
  if( is_cmplx ){ 
    fi = mxGetPi(plhs[0]); 
    yi = mxGetPi(plhs[1]);  
  }

  /* call algorithm subroutines */
  switch( alg_num ){
    case CMA: 
      D_u = (int)mxGetScalar(prhs[10]);
      cma(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y, D_u); break;
    case NCMA: 
      alpha = (double)mxGetScalar(prhs[10]);
      ncma(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y, alpha); break;
    case SECMA:
      secma(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y); break;
    case SRCMA:
      srcma(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y); break;
    case SSCMA:
      sscma(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y); break;
    case DSECMA:
      alpha = (double)mxGetScalar(prhs[10]);
      dsecma(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y, alpha); break;
    case WSGA:
      wsga(fr, fi, yr, yi, srcme, rr, ri, fr_init, fi_init, kappa, 
     	  mu, N, N_f, D_e, D_f, is_FSE, is_cmplx, len_f, len_y); break;
    case CMAGD:
      mexErrMsgTxt("berg_algs.mexlx does not support CMA-GD."); break;
    default:
      mexErrMsgTxt("berg_algs.mexlx : You shouldn't be here!");
  }
}



