clear;
close all;
%-----------------------------------------------------------------------
% Matlab implementation of the algorithms described in IEEE TRANSACTIONS ON COMPUTATIONAL IMAGING, VOL. 1, NO. 4, DECEMBER 2015 
% Coded by: Rizwan Ahmad (Adhmad.46@osu.edu)
% Last modified: Feb. 17, 2017
%-----------------------------------------------------------------------

% Add relevant folders to the path
addpath('.\data');
addpath('.\utility');
addpath('.\UWT');

ex = 4; % which of the four examples to run

if ex == 1
    % Example 1 -- finite difference example
    x = getAnisoImage2D([48,48], 1, [25,3]); 
    figure; imagesc(abs(x)); axis('image'); colormap(gray);
    p = paramDifference(x); % upload parameters; change SNR, sampling ratio here

elseif ex ==2 
    % Example 2 -- MRI example
    x = double(importdata('MRI2D.mat'));
    figure; imagesc(abs(x)); axis('image'); colormap(gray);
    p = paramMRI(x); % upload parameters; change SNR, sampling ratio here

elseif ex == 3
    % Example 3 -- cameraman example
    x = im2double(imread('cameraman_256.tiff')); 
    x = x(37:end-124,67:end-86); % Crop to reduce processing time
    figure; imagesc(abs(x)); axis('image'); colormap(gray);
    p = paramCameraman(x); % upload parameters; change SNR, sampling ratio here

elseif ex ==4
    % Example 4 -- complex-valued shepp-logan phantom example
    x = phantom('modified shepp-logan', 96);
    x = x - mean(x(:));
    x = x + 1j*x;
    figure; imagesc(abs(x)); axis('image'); colormap(gray);
    p = paramShepplogan(x); % upload parameters; change SNR, sampling ratio here
end


%% Place holders for results
p.rmse = nan(4,1);      % store recovery rmse here
xHat   = nan([4,p.n]);  % store recovered images

%% build operators;
[p.A, p.At] = sensingMatrix(p, p.acce); % Sensing matrix and its adjoint
[p.U, p.Ut, p.bInd] = wavWrap(x, p); % Sparsifying transform and its adjoint

%% Simuate data and add noise
[y, p.nStd, sclFctr] = genNoisyData(p,x);

%% Initialization
x0   = p.At(y); % Initialization for x
tmpCoef = p.U(x0);
p.lmb0  = 0.25/mean(abs(tmpCoef(:)));   % Initial \lambda guess
p.epn0  = 1e-1*median(abs(tmpCoef(:))); % Initial \epsilon guess
clear tmpCoef;

%% FISTA, L1
disp('========== Running L1 ==========');
% Generaized L1, with auto-tuning of one global \lambda
p.rw   = 0; 
p.adap = 1; 
[tmp, ~, ~] = mFST_Rwl1(y, x0, p);
xHat(1,:,:) = 1/sclFctr * tmp;
p.rmse(1) = norm(x(:)-xHat(1,:).')/norm(x(:));

%% FISTA, L1 Adaptive
disp('========== Running Co-L1 ==========');
% Generaized L1, with per subband auto-tuning of \lambda
p.rw   = 0; 
p.adap = 2; 
[tmp, ~,~] = mFST_Rwl1(y, x0, p);
xHat(2,:,:) = 1/sclFctr * tmp;
p.rmse(2) = norm(x(:)-xHat(2,:).')/norm(x(:));    

%% FISTA, RWL1
disp('========== Running IRW-L1 ==========');
% Iteratively reweightd L1 with auto-tuning of one global \lambda and \epsilon
p.rw   = 1; 
p.adap = 1; 
[tmp, tmpLmb, tmpG]  =  mFST_Rwl1(y, x0, p);
xHat(3,:,:) = 1/sclFctr * tmp;
p.rmse(3) = norm(x(:)-xHat(3,:).')/norm(x(:));

%% FISTA, RwL1 Adaptive
disp('========== Running Co-IRW-L1 ==========');
% Iteratively reweightd L1 with per subband auto-tuning of \lambda and \epsilon
p.rw   = 1; 
p.adap = 2; 
[tmp,~,~] = mFST_Rwl1(y, x0, p);
xHat(4,:,:) = 1/sclFctr * tmp;
p.rmse(4) = norm(x(:)-xHat(4,:).')/norm(x(:));
% 20*log10(p.rmse)

%% Ploting figures
figure;
if isreal(x)
    mn = min(x(:));
    mx = max(x(:));
else
    mn = min(abs(x(:)));
    mx = max(abs(x(:)));
end

subplot(1,3,1); imagesc(real(squeeze(x)),[mn,mx]); axis('image','off'); colormap('gray');
title('noiseless input');
subplot(2,3,2); imagesc(real(squeeze(xHat(1,:,:))),[mn,mx]); axis('image','off'); colormap('gray');
title(['Auto-tuned L1, SNR: '     num2str(-20*log10(p.rmse(1)),3),'dB']);
subplot(2,3,3); imagesc(real(squeeze(xHat(2,:,:))),[mn,mx]); axis('image','off'); colormap('gray');
title(['Auto-tuned Co-L1, SNR: '  num2str(-20*log10(p.rmse(2)),3),'dB']);
subplot(2,3,5); imagesc(real(squeeze(xHat(3,:,:))),[mn,mx]); axis('image','off'); colormap('gray');
title(['Auto-tuned RWL1, SNR: '   num2str(-20*log10(p.rmse(3)),3),'dB']);
subplot(2,3,6); imagesc(real(squeeze(xHat(4,:,:))),[mn,mx]); axis('image','off'); colormap('gray');
title(['Auto-tuned Co-RWL1, SNR: ' num2str(-20*log10(p.rmse(4)),3),'dB']);
